/**
 * Stone of Orthanc
 * Copyright (C) 2012-2016 Sebastien Jodogne, Medical Physics
 * Department, University Hospital of Liege, Belgium
 * Copyright (C) 2017-2023 Osimis S.A., Belgium
 * Copyright (C) 2021-2025 Sebastien Jodogne, ICTEAM UCLouvain, Belgium
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program. If not, see
 * <http://www.gnu.org/licenses/>.
 **/


#include "../Sources/Toolbox/PixelTestPatterns.h"

#include <Images/Image.h>
#include <Images/PngWriter.h>

#include <boost/chrono.hpp>
#include <boost/lexical_cast.hpp>

#include <gtest/gtest.h>
#include <stdint.h>
#include <cmath>

 /* Autogenerated from prout.png */
static const unsigned char bin2c_SimpleRedBlueHGradient_png[391] = "\211PNG\15\12\32\12\0\0\0\15IHDR\0\0\0\200\0\0\0\200\10\6\0\0\0\303>a\313\0\0\1MIDATx\234\355\322\1\15\3000\0\303\260~\3741o\7\342X\12\203|o{wgev\26Z\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p\15\200k\0\\\3\340\32\0\327\0\270\6\3005\0\256\1p?\314\262\201\3760\355r\262\0\0\0\0IEND\256B`\202";

TEST(PixelTestPatterns, SimpleRedHGradient)
{
  std::unique_ptr<Orthanc::Image> texture;

  texture.reset(new Orthanc::Image(
    Orthanc::PixelFormat_RGBA32,
    128,
    128,
    /*forceMinimalPitch*/false));

  Orthanc::ImageAccessor target;
  texture->GetWriteableAccessor(target);

  OrthancStone::PixelTestPatterns::fillWithHGradient(target,255,0,0,0,0,255);

  Orthanc::PngWriter writer;
#if 0
  Orthanc::IImageWriter::WriteToFile(writer, "SimpleRedBlueHGradient.png", *texture);
#else
  std::string contents;
  Orthanc::IImageWriter::WriteToMemory(writer, contents, *texture);

  ASSERT_EQ(1u, sizeof(unsigned char));
  ASSERT_EQ(391u, sizeof(bin2c_SimpleRedBlueHGradient_png));
  ASSERT_EQ(390u, contents.size());

  const char* resultPngBytes = &(contents[0]);

  int result = memcmp(resultPngBytes, bin2c_SimpleRedBlueHGradient_png, 390);
  ASSERT_EQ(0, result);
#endif
}

static const unsigned char bin2c_SimpleRedBlueVGradient_png[400] = "\211PNG\15\12\32\12\0\0\0\15IHDR\0\0\0\200\0\0\0\200\10\6\0\0\0\303>a\313\0\0\1VIDATx\234\355\322A\21\3000\14\300\260t7\376\220\327\301\310\303\22\2?|\356\314\35\262\336o\236\355\6\26\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\14\20g\2008\3\304\31 \316\0q\6\2103@\234\1\342\316\231\357nG\260\347\7\221\255\203\367~A)\36\0\0\0\0IEND\256B`\202";


TEST(PixelTestPatterns, SimpleRedBlueVGradient)
{
  std::unique_ptr<Orthanc::Image> texture;

  texture.reset(new Orthanc::Image(
    Orthanc::PixelFormat_RGBA32,
    128,
    128,
    /*forceMinimalPitch*/false));

  Orthanc::ImageAccessor target;
  texture->GetWriteableAccessor(target);

  OrthancStone::PixelTestPatterns::fillWithVGradient(target, 255, 0, 0, 0, 0, 255);

  Orthanc::PngWriter writer;
#if 0
  Orthanc::IImageWriter::WriteToFile(writer, "SimpleRedBlueVGradient.png", *texture);
#else
  std::string contents;
  Orthanc::IImageWriter::WriteToMemory(writer, contents, *texture);

  ASSERT_EQ(1u, sizeof(unsigned char));
  ASSERT_EQ(400u, sizeof(bin2c_SimpleRedBlueVGradient_png));
  ASSERT_EQ(399u, contents.size());

  const char* resultPngBytes = &(contents[0]);

  int result = memcmp(resultPngBytes, bin2c_SimpleRedBlueVGradient_png, 399);
  ASSERT_EQ(0, result);
#endif
}


/* Autogenerated from MultiGradient.png */
static const unsigned char bin2c_MultiGradient_png[774] = "\211PNG\15\12\32\12\0\0\0\15IHDR\0\0\1\0\0\0\0\200\10\6\0\0\0\344\265\267\12\0\0\2\314IDATx\234\355\325\301\11\3030\24\5\301/\242\376+6(E$ \314\354\30\337\215\364X\2573s\6\266\316\314\226\237\365\271}\5\227\255\331{\273\357s\373\374sY\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\25\0^\13\220\2559\347\354\231Q\337\317\372\303)\276\331Ys\377\26\256.\340\3673|\261}\373\3n{\360?\240>\200\347\351\376i\5\300V\0pz\0t\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0l\5\0W\0lz\0\276!\352\302\35+U\244b\0\0\0\0IEND\256B`\202";

TEST(PixelTestPatterns, MultiGradient)
{
  std::unique_ptr<Orthanc::Image> texture;

  const int CELLW = 64;
  const int CELLH = 64;
  const int NHCELLS = 4;
  const int NVCELLS = 2;
  const int NCELLS = NHCELLS * NVCELLS;

  texture.reset(new Orthanc::Image(
    Orthanc::PixelFormat_RGBA32,
    NHCELLS * CELLW,
    NVCELLS * CELLH,
    /*forceMinimalPitch*/false));

  // H:R->K, V:G->W, H:B->K

  //                              R   G   B   K   C   M   Y   W
  const uint8_t startR[NCELLS] = {255,000,000,000,000,255,255,255};
  const uint8_t startG[NCELLS] = {000,255,000,000,255,000,255,255};
  const uint8_t startB[NCELLS] = {000,000,255,000,255,255,000,255};
  
  //                              K   W   K   W   W   K   W   K
  const uint8_t eeendR[NCELLS] = {000,255,000,255,255,000,255,000};
  const uint8_t eeendG[NCELLS] = {000,255,000,255,255,000,255,000 };
  const uint8_t eeendB[NCELLS] = {000,255,000,255,255,000,255,000 };

  for(size_t slot = 0; slot < NCELLS; ++slot)
  {
    int x0 = (slot % 4) * CELLW;
    bool vertical = (((slot / NHCELLS) % 2) == 0) ? (slot % 2 == 0) : (slot % 2 == 1);
    int y0 = static_cast<int>(slot / NHCELLS) * CELLH;
    Orthanc::ImageAccessor target;
    texture->GetRegion(target, x0, y0, CELLW, CELLH);
    if (vertical)
      OrthancStone::PixelTestPatterns::fillWithVGradient(target, startR[slot], startG[slot], startB[slot], eeendR[slot], eeendG[slot], eeendB[slot]);
    else 
      OrthancStone::PixelTestPatterns::fillWithHGradient(target, startR[slot], startG[slot], startB[slot], eeendR[slot], eeendG[slot], eeendB[slot]);
  }
 
  Orthanc::PngWriter writer;
#if 0
  Orthanc::IImageWriter::WriteToFile(writer, "MultiGradient.png", *texture);
#else
  std::string contents;
  Orthanc::IImageWriter::WriteToMemory(writer, contents, *texture);

  ASSERT_EQ(1u, sizeof(unsigned char));
  ASSERT_EQ(774u, sizeof(bin2c_MultiGradient_png));
  ASSERT_EQ(773u, contents.size());

  const char* resultPngBytes = &(contents[0]);

  int result = memcmp(resultPngBytes, bin2c_MultiGradient_png, 773);
  ASSERT_EQ(0, result);
#endif
}

