/**
 * Stone of Orthanc
 * Copyright (C) 2012-2016 Sebastien Jodogne, Medical Physics
 * Department, University Hospital of Liege, Belgium
 * Copyright (C) 2017-2023 Osimis S.A., Belgium
 * Copyright (C) 2021-2025 Sebastien Jodogne, ICTEAM UCLouvain, Belgium
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program. If not, see
 * <http://www.gnu.org/licenses/>.
 **/


#pragma once

#include "../../OrthancStone.h"

#if !defined(ORTHANC_ENABLE_WASM)
#  error Macro ORTHANC_ENABLE_WASM must be defined
#endif

#if ORTHANC_ENABLE_WASM != 1
#  error This file can only be used if targeting WebAssembly
#endif

#include "../../Viewport/IViewport.h"
#include "../../Viewport/IViewportInteractor.h"

#include <Compatibility.h>

#include <emscripten.h>
#include <emscripten/html5.h>

#include <memory>
#include <string>
#include <vector>
#include <boost/enable_shared_from_this.hpp>

namespace OrthancStone
{
  class WebAssemblyViewport : public IViewport,
                              public boost::enable_shared_from_this<WebAssemblyViewport>

  {
  private:
    class WasmLock;
    
    std::string                           canvasId_;
    std::string                           canvasCssSelector_;
    std::unique_ptr<ICompositor>          compositor_;
    std::unique_ptr<ViewportController>   controller_;
    std::unique_ptr<IViewportInteractor>  interactor_;
    bool                                  enableEmscriptenMouseEvents_;
    unsigned int                          canvasWidth_;
    unsigned int                          canvasHeight_;

    static EM_BOOL OnRequestAnimationFrame(double time, void *userData);
    
    static EM_BOOL OnResize(int eventType, const EmscriptenUiEvent *uiEvent, void *userData);

    static EM_BOOL OnMouseDown(int eventType, const EmscriptenMouseEvent *mouseEvent, void *userData);
    
    static EM_BOOL OnMouseMove(int eventType, const EmscriptenMouseEvent *mouseEvent, void *userData);
    
    static EM_BOOL OnMouseUp(int eventType, const EmscriptenMouseEvent *mouseEvent, void *userData);

    static EM_BOOL OnTouch(int eventType, const EmscriptenTouchEvent *mouseEvent, void *userData);

  protected:
    void Invalidate();
    
    void ClearCompositor()
    {
      compositor_.reset();
    }

    bool HasCompositor() const
    {
      return compositor_.get() != NULL;
    }

    void AcquireCompositor(ICompositor* compositor /* takes ownership */);

    virtual void Paint(ICompositor& compositor,
                       ViewportController& controller) = 0;

    /**
    The second argument is temporary and should be deleted once the migration 
    to interactors is finished. It should be set to "true" for new applications.
    */
    WebAssemblyViewport(const std::string& canvasId, 
                        bool enableEmscriptenMouseEvents);

    void PostConstructor();


    /**
     * This method can be called to retrieve a cookie that can be passed to 
     * C-style callbacks that expect the object to be passed as a void*
     * 
     * This cookie is a resource and must be freed when it is guaranteed 
     * not to be used anymore, with ReleaseObjectCookie
     */
    void* CreateObjectCookie();

    /**
     * This static method can be used to dereference a cookie (i.e. retrieve 
     * a pointer to the underlying object) that has been created with 
     * WebAssemblyViewport::CreateObjectCookie()
     * 
     * If this method returns NULL, it basically means that the 
     * WebAssemblyViewport has already been deleted and that you should NOT 
     * attempt to use it!
     * 
     * This method never releases the cookie (for other in-flight callbacks
     * could possibly require to cookie to be valid)
     * 
     * If this method is called AFTER ReleaseObjectCookie has been called on
     * the same cookie, the behavior is undefined and things will CERTAINLY
     * go wrong.
     * 
     * NEVER call this method on a void* that has not been generated by the
     * CreateObjectCookie method of this class
     */
    static WebAssemblyViewport* DereferenceObjectCookie(void* cookie);

    /**
     * This method must be used when the object cookie will not be used anymore.
     * 
     * You must call it when you are certain that no entity will attempt to 
     * dereference the cookie.
     */
    static void ReleaseObjectCookie(void* cookie);

    void RefreshCanvasSize();

    unsigned int GetCanvasWidth() const
    {
      return canvasWidth_;
    }
    
    unsigned int GetCanvasHeight()
    {
      return canvasHeight_;
    }

  public:
    virtual ILock* Lock() ORTHANC_OVERRIDE;

    ~WebAssemblyViewport();

    /**
    This method takes ownership
    */
    void AcquireInteractor(IViewportInteractor* interactor);

    const std::string& GetCanvasId() const
    {
      return canvasId_;
    }

    /**
    emscripten functions requires the css selector for the canvas. This is 
    different from the canvas id (the syntax is '#mycanvasid')
    */
    const std::string& GetCanvasCssSelector() const
    {
      return canvasCssSelector_;
    }

    void FitForPrint();
  };
}
