
#define BGO_ENABLE_DICOMSTRUCTURESETLOADER2

// working around a bug where the Visual C++ compiler would get 
// stuck trying to compile this cpp file in release mode
// (versions: https://en.wikipedia.org/wiki/Microsoft_Visual_C%2B%2B)
#ifdef _MSC_VER
#  pragma optimize("", off)
// warning C4748: /GS can not protect parameters and local variables from
// local buffer overrun because optimizations are disabled in function
#  pragma warning(disable: 4748)
#endif



#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/make_shared.hpp>
#include <string>



using namespace OrthancStone;

static const double DELTA_MAX = 10.0 * std::numeric_limits<float>::epsilon();



#define STONE_ARRAY_SIZE(arr)     (sizeof(arr) / sizeof((arr)[0]))

static double pointsCoord1[] = { 2, 2, 3,  3,  6,  8, 8, 7, 8, 8, 6 };
static double pointsCoord2[] = { 2, 6, 8, 10, 12, 10, 8, 6, 4, 2, 4 };
static const size_t pointsCoord1Count = STONE_ARRAY_SIZE(pointsCoord1);
static const size_t pointsCoord2Count = STONE_ARRAY_SIZE(pointsCoord2);
const size_t POLYGON_POINT_COUNT = pointsCoord1Count;


#ifdef BGO_ENABLE_DICOMSTRUCTURESETLOADER2

static void CheckGroundTruth(
  const std::vector<DicomStructure2>& structures,
  const size_t structureIndex,
  const size_t sliceIndex,
  std::vector<double> groundTruth)
{
  const std::vector<DicomStructurePolygon2>& polygonsForThisStruct = structures.at(structureIndex).GetPolygons();
  const DicomStructurePolygon2& polygon = polygonsForThisStruct.at(sliceIndex);

  //double groundTruth[] = { 7.657838, 108.2725, 304.01, 6.826687, 107.4413, 304.01, 6.152492, 106.4785, 304.01, 5.655735, 105.4132, 304.01, 5.351513, 104.2778, 304.01, 5.249068, 103.1069, 304.01, 5.351513, 101.9359, 304.01, 5.655735, 100.8005, 304.01, 6.152492, 99.73524, 304.01, 6.826687, 98.77239, 304.01, 7.657838, 97.94124, 304.01, 8.620689, 97.26704, 304.01, 9.685987, 96.77029, 304.01, 10.82136, 96.46606, 304.01, 11.99231, 96.36362, 304.01, 13.16326, 96.46606, 304.01, 14.29864, 96.77029, 304.01, 15.36393, 97.26704, 304.01, 16.32678, 97.94124, 304.01, 17.15794, 98.77239, 304.01, 17.83213, 99.73524, 304.01, 18.32889, 100.8005, 304.01, 18.63311, 101.9359, 304.01, 18.73555, 103.1069, 304.01, 18.63311, 104.2778, 304.01, 18.32889, 105.4132, 304.01, 17.83213, 106.4785, 304.01, 17.15794, 107.4413, 304.01, 16.32678, 108.2725, 304.01, 15.36393, 108.9467, 304.01, 14.29864, 109.4434, 304.01, 13.16326, 109.7477, 304.01, 11.99231, 109.8501, 304.01, 10.82136, 109.7477, 304.01, 9.685987, 109.4434, 304.01, 8.620689, 108.9467, 304.01 };
  size_t groundTruthItems = groundTruth.size();

  size_t pointCount = 3 * polygon.GetPointCount();

  EXPECT_EQ(groundTruthItems, pointCount);

  for (size_t i = 0; i < polygon.GetPointCount(); ++i)
  {
    const Vector& point = polygon.GetPoint(i);

    // loop over X, Y then Z.
    for (size_t j = 0; j < 3; ++j)
    {
      size_t index = 3 * i + j;
      ASSERT_LT(index, groundTruthItems);
      bool isNear = LinearAlgebra::IsNear(groundTruth[index], point[j]);
      EXPECT_TRUE(isNear);
    }
  }
}


#include <Toolbox.h>

TEST(StructureSet2, ReadFromJsonThatsAll)
{
  /*
    The "RT_STRUCT_00" string is the reply to the following Orthanc request:
    
    http://localhost:8042/instances/1aa5f84b-c32a03b4-3c1857da-da2e69f3-3ef6e2b3/tags?ignore-length=3006-0050
    
    The tag hierarchy can be found here: https://dicom.innolitics.com/ciods/rt-dose
  */

  DicomStructureSet2 structureSet;

  FullOrthancDataset dicom(Orthanc::EmbeddedResources::GetFileResourceBuffer(Orthanc::EmbeddedResources::RT_STRUCT_00),
                           Orthanc::EmbeddedResources::GetFileResourceSize(Orthanc::EmbeddedResources::RT_STRUCT_00));
  structureSet.Clear();

  structureSet.FillStructuresFromDataset(dicom);
  structureSet.ComputeDependentProperties();

  const std::vector<DicomStructure2>& structures = structureSet.structures_;

  /*

  ██████╗  █████╗ ███████╗██╗ ██████╗     ██████╗██╗  ██╗███████╗ ██████╗██╗  ██╗███████╗
  ██╔══██╗██╔══██╗██╔════╝██║██╔════╝    ██╔════╝██║  ██║██╔════╝██╔════╝██║ ██╔╝██╔════╝
  ██████╔╝███████║███████╗██║██║         ██║     ███████║█████╗  ██║     █████╔╝ ███████╗
  ██╔══██╗██╔══██║╚════██║██║██║         ██║     ██╔══██║██╔══╝  ██║     ██╔═██╗ ╚════██║
  ██████╔╝██║  ██║███████║██║╚██████╗    ╚██████╗██║  ██║███████╗╚██████╗██║  ██╗███████║
  ╚═════╝ ╚═╝  ╚═╝╚══════╝╚═╝ ╚═════╝     ╚═════╝╚═╝  ╚═╝╚══════╝ ╚═════╝╚═╝  ╚═╝╚══════╝
  http://patorjk.com/software/taag/#p=display&f=ANSI%20Shadow&t=BASIC%20CHECKS
  */

  // (0x3006, 0x0080) seq. size
  EXPECT_EQ(7u, structures.size());

  // (0x3006, 0x0080)[i]/(0x3006, 0x00a4)
  for (size_t i = 0; i < 5; ++i)
  {
    EXPECT_EQ(std::string("ORGAN"), structures[i].interpretation_);
  }
  EXPECT_EQ(std::string("EXTERNAL"), structures[5].interpretation_);
  EXPECT_EQ(std::string("PTV"), structures[6].interpretation_);

  // (0x3006, 0x0020)[i]/(0x3006, 0x0026)
  EXPECT_EQ(std::string("LN300"), structures[0].name_);
  EXPECT_EQ(std::string("Cortical Bone"), structures[1].name_);
  EXPECT_EQ(std::string("Adipose"), structures[2].name_);
  EXPECT_EQ(std::string("CB2-50%"), structures[3].name_);
  EXPECT_EQ(std::string("Water"), structures[4].name_);
  EXPECT_EQ(std::string("External"), structures[5].name_);
  EXPECT_EQ(std::string("PTV"), structures[6].name_);

  // (0x3006, 0x0039)[i]/(0x3006, 0x002a)
  EXPECT_EQ(0xff, structures[0].red_);
  EXPECT_EQ(0x00, structures[0].green_);
  EXPECT_EQ(0x00, structures[0].blue_);

  EXPECT_EQ(0x00, structures[1].red_);
  EXPECT_EQ(0xff, structures[1].green_);
  EXPECT_EQ(0xff, structures[1].blue_);

  // ...

  EXPECT_EQ(0x00, structures[5].red_);
  EXPECT_EQ(0x80, structures[5].green_);
  EXPECT_EQ(0x00, structures[5].blue_);

  EXPECT_EQ(0xff, structures[6].red_);
  EXPECT_EQ(0x00, structures[6].green_);
  EXPECT_EQ(0xff, structures[6].blue_);

  /*

   ██████╗ ███████╗ ██████╗ ███╗   ███╗███████╗████████╗██████╗ ██╗   ██╗
  ██╔════╝ ██╔════╝██╔═══██╗████╗ ████║██╔════╝╚══██╔══╝██╔══██╗╚██╗ ██╔╝
  ██║  ███╗█████╗  ██║   ██║██╔████╔██║█████╗     ██║   ██████╔╝ ╚████╔╝
  ██║   ██║██╔══╝  ██║   ██║██║╚██╔╝██║██╔══╝     ██║   ██╔══██╗  ╚██╔╝
  ╚██████╔╝███████╗╚██████╔╝██║ ╚═╝ ██║███████╗   ██║   ██║  ██║   ██║
   ╚═════╝ ╚══════╝ ╚═════╝ ╚═╝     ╚═╝╚══════╝   ╚═╝   ╚═╝  ╚═╝   ╚═╝
  http://patorjk.com/software/taag/#p=display&f=ANSI%20Shadow&t=BASIC%20CHECKS
  */


  {
    double groundTruthRaw[] = { 7.657838, 108.2725, 304.01, 6.826687, 107.4413, 304.01, 6.152492, 106.4785, 304.01, 5.655735, 105.4132, 304.01, 5.351513, 104.2778, 304.01, 5.249068, 103.1069, 304.01, 5.351513, 101.9359, 304.01, 5.655735, 100.8005, 304.01, 6.152492, 99.73524, 304.01, 6.826687, 98.77239, 304.01, 7.657838, 97.94124, 304.01, 8.620689, 97.26704, 304.01, 9.685987, 96.77029, 304.01, 10.82136, 96.46606, 304.01, 11.99231, 96.36362, 304.01, 13.16326, 96.46606, 304.01, 14.29864, 96.77029, 304.01, 15.36393, 97.26704, 304.01, 16.32678, 97.94124, 304.01, 17.15794, 98.77239, 304.01, 17.83213, 99.73524, 304.01, 18.32889, 100.8005, 304.01, 18.63311, 101.9359, 304.01, 18.73555, 103.1069, 304.01, 18.63311, 104.2778, 304.01, 18.32889, 105.4132, 304.01, 17.83213, 106.4785, 304.01, 17.15794, 107.4413, 304.01, 16.32678, 108.2725, 304.01, 15.36393, 108.9467, 304.01, 14.29864, 109.4434, 304.01, 13.16326, 109.7477, 304.01, 11.99231, 109.8501, 304.01, 10.82136, 109.7477, 304.01, 9.685987, 109.4434, 304.01, 8.620689, 108.9467, 304.01 };
    size_t n = sizeof(groundTruthRaw) / sizeof(groundTruthRaw[0]); 
    std::vector<double> groundTruth(groundTruthRaw, groundTruthRaw+n);
    CheckGroundTruth(structures, 0, 0, groundTruth);
  }
  {
    double groundTruthRaw[] = { 7.657838, 108.2725, 310.01, 6.826687, 107.4413, 310.01, 6.152492, 106.4785, 310.01, 5.655735, 105.4132, 310.01, 5.351513, 104.2778, 310.01, 5.249068, 103.1069, 310.01, 5.351513, 101.9359, 310.01, 5.655735, 100.8005, 310.01, 6.152492, 99.73524, 310.01, 6.826687, 98.77239, 310.01, 7.657838, 97.94124, 310.01, 8.620689, 97.26704, 310.01, 9.685987, 96.77029, 310.01, 10.82136, 96.46606, 310.01, 11.99231, 96.36362, 310.01, 13.16326, 96.46606, 310.01, 14.29864, 96.77029, 310.01, 15.36393, 97.26704, 310.01, 16.32678, 97.94124, 310.01, 17.15794, 98.77239, 310.01, 17.83213, 99.73524, 310.01, 18.32889, 100.8005, 310.01, 18.63311, 101.9359, 310.01, 18.73555, 103.1069, 310.01, 18.63311, 104.2778, 310.01, 18.32889, 105.4132, 310.01, 17.83213, 106.4785, 310.01, 17.15794, 107.4413, 310.01, 16.32678, 108.2725, 310.01, 15.36393, 108.9467, 310.01, 14.29864, 109.4434, 310.01, 13.16326, 109.7477, 310.01, 11.99231, 109.8501, 310.01, 10.82136, 109.7477, 310.01, 9.685987, 109.4434, 310.01, 8.620689, 108.9467, 310.01 };
    size_t n = sizeof(groundTruthRaw) / sizeof(groundTruthRaw[0]); 
    std::vector<double> groundTruth(groundTruthRaw, groundTruthRaw+n);
    CheckGroundTruth(structures, 0, 2, groundTruth);
  }
  {
    double groundTruthRaw[] = { -37.967, 161.9664, 304.01, -39.10237, 161.6622, 304.01, -40.16767, 161.1655, 304.01, -41.13052, 160.4913, 304.01, -41.96167, 159.6601, 304.01, -42.63587, 158.6973, 304.01, -43.13263, 157.632, 304.01, -43.43685, 156.4966, 304.01, -43.53929, 155.3257, 304.01, -43.43685, 154.1547, 304.01, -43.13263, 153.0193, 304.01, -42.63587, 151.954, 304.01, -41.96167, 150.9912, 304.01, -41.13052, 150.16, 304.01, -40.16767, 149.4858, 304.01, -39.10237, 148.9891, 304.01, -37.967, 148.6849, 304.01, -36.79605, 148.5824, 304.01, -35.6251, 148.6849, 304.01, -34.48972, 148.9891, 304.01, -33.42443, 149.4858, 304.01, -32.46157, 150.16, 304.01, -31.63042, 150.9912, 304.01, -30.95623, 151.954, 304.01, -30.45947, 153.0193, 304.01, -30.15525, 154.1547, 304.01, -30.0528, 155.3257, 304.01, -30.15525, 156.4966, 304.01, -30.45947, 157.632, 304.01, -30.95623, 158.6973, 304.01, -31.63042, 159.6601, 304.01, -32.46157, 160.4913, 304.01, -33.42443, 161.1655, 304.01, -34.48972, 161.6622, 304.01, -35.6251, 161.9664, 304.01, -36.79605, 162.0689, 304.01 };
    size_t n = sizeof(groundTruthRaw) / sizeof(groundTruthRaw[0]); 
    std::vector<double> groundTruth(groundTruthRaw, groundTruthRaw+n);
    CheckGroundTruth(structures, 1, 0, groundTruth);
  }
  {
    double groundTruthRaw[] = { 69.4042, 150.7324, 307.01, 69.70842, 151.8678, 307.01, 69.81087, 153.0387, 307.01, 69.70842, 154.2097, 307.01, 69.4042, 155.345, 307.01, 68.90745, 156.4103, 307.01, 68.23325, 157.3732, 307.01, 67.4021, 158.2043, 307.01, 66.43925, 158.8785, 307.01, 65.37395, 159.3753, 307.01, 64.23858, 159.6795, 307.01, 63.06762, 159.7819, 307.01, 61.89667, 159.6795, 307.01, 60.7613, 159.3753, 307.01, 59.696, 158.8785, 307.01, 58.73315, 158.2043, 307.01, 57.902, 157.3732, 307.01, 57.22781, 156.4103, 307.01, 56.73105, 155.345, 307.01, 56.42683, 154.2097, 307.01, 56.32438, 153.0387, 307.01, 56.42683, 151.8678, 307.01, 56.73105, 150.7324, 307.01, 57.22781, 149.6671, 307.01, 57.902, 148.7042, 307.01, 58.73315, 147.8731, 307.01, 59.696, 147.1989, 307.01, 60.7613, 146.7021, 307.01, 61.89667, 146.3979, 307.01, 63.06762, 146.2955, 307.01, 64.23858, 146.3979, 307.01, 65.37395, 146.7021, 307.01, 66.43925, 147.1989, 307.01, 67.4021, 147.8731, 307.01, 68.23325, 148.7042, 307.01, 68.90745, 149.6671, 307.01 };
    size_t n = sizeof(groundTruthRaw) / sizeof(groundTruthRaw[0]); 
    std::vector<double> groundTruth(groundTruthRaw, groundTruthRaw+n);
    CheckGroundTruth(structures, 2, 1, groundTruth);
  }

  {
    double groundTruthRaw[] = { 108.3984, 232.7406, 274.01, 106.0547, 231.7948, 274.01, 103.7109, 232.8407, 274.01, 96.67969, 232.8757, 274.01, 77.92969, 232.887, 274.01, 47.46094, 232.8902, 274.01, 38.08594, 232.7537, 274.01, 37.6668, 232.3734, 274.01, 38.08594, 231.9774, 274.01, 40.42969, 231.8475, 274.01, 41.76413, 230.0297, 274.01, 42.77344, 229.1388, 274.01, 45.11719, 228.5069, 274.01, 47.46094, 227.1533, 274.01, 49.80469, 226.3505, 274.01, 52.14844, 224.6564, 274.01, 54.49219, 223.923, 274.01, 56.83594, 222.0692, 274.01, 59.17969, 220.3438, 274.01, 61.52344, 219.3888, 274.01, 63.86719, 217.1287, 274.01, 65.83488, 215.9672, 274.01, 68.55469, 213.2383, 274.01, 70.89844, 211.2328, 274.01, 72.8125, 208.9359, 274.01, 75.58594, 206.3615, 274.01, 76.91445, 204.2484, 274.01, 78.89509, 201.9047, 274.01, 80.51276, 199.5609, 274.01, 81.51955, 197.2172, 274.01, 83.67448, 194.8734, 274.01, 84.60938, 192.5297, 274.01, 85.86986, 190.1859, 274.01, 86.57623, 187.8422, 274.01, 88.30051, 185.4984, 274.01, 88.94002, 183.1547, 274.01, 89.23261, 180.8109, 274.01, 89.64844, 180.3263, 274.01, 90.71885, 178.4672, 274.01, 90.97656, 176.1234, 274.01, 91.99219, 174.4794, 274.01, 92.56773, 173.7797, 274.01, 92.80016, 171.4359, 274.01, 93.23473, 169.0922, 274.01, 93.37606, 166.7484, 274.01, 93.60748, 157.3734, 274.01, 93.6341, 152.6859, 274.01, 93.35742, 140.9672, 274.01, 92.89317, 138.6234, 274.01, 92.7069, 136.2797, 274.01, 92.03726, 133.9359, 274.01, 90.84009, 131.5922, 274.01, 90.3769, 129.2484, 274.01, 89.09074, 126.9047, 274.01, 88.13225, 122.2172, 274.01, 86.17828, 119.8734, 274.01, 84.96094, 117.4163, 274.01, 83.99619, 115.1859, 274.01, 83.13079, 112.8422, 274.01, 82.61719, 112.2984, 274.01, 80.27344, 108.8454, 274.01, 79.64514, 108.1547, 274.01, 77.21497, 105.8109, 274.01, 76.47787, 103.4672, 274.01, 75.58594, 102.6177, 274.01, 73.24219, 100.0077, 274.01, 69.54492, 96.43594, 274.01, 67.34096, 94.09219, 274.01, 64.66306, 91.74844, 274.01, 63.86719, 90.92619, 274.01, 61.52344, 90.20454, 274.01, 59.17969, 87.78574, 274.01, 56.83594, 86.48566, 274.01, 54.49219, 84.31388, 274.01, 52.14844, 83.44438, 274.01, 49.80469, 82.75121, 274.01, 49.37617, 82.37344, 274.01, 47.46094, 81.26244, 274.01, 45.71391, 80.02969, 274.01, 45.11719, 79.45415, 274.01, 42.77344, 79.08185, 274.01, 40.42969, 78.51941, 274.01, 38.08594, 78.27534, 274.01, 37.36932, 77.68594, 274.01, 35.74219, 76.67624, 274.01, 33.39844, 76.49941, 274.01, 31.05469, 76.03495, 274.01, 28.71094, 74.83174, 274.01, 26.36719, 74.62859, 274.01, 24.02344, 74.55463, 274.01, 21.67969, 74.22861, 274.01, 19.33594, 74.05312, 274.01, 12.30469, 73.99397, 274.01, 5.273438, 74.0736, 274.01, 2.929688, 74.55463, 274.01, 0.5859375, 74.68513, 274.01, -1.757813, 74.914, 274.01, -2.319131, 75.34219, 274.01, -4.101563, 76.31516, 274.01, -8.789063, 76.74514, 274.01, -11.13281, 78.39038, 274.01, -13.47656, 78.6124, 274.01, -15.82031, 79.19784, 274.01, -18.16406, 81.11024, 274.01, -20.50781, 82.03296, 274.01, -22.85156, 83.13991, 274.01, -25.19531, 83.70732, 274.01, -27.53906, 85.85863, 274.01, -29.88281, 87.03368, 274.01, -32.22656, 88.3274, 274.01, -34.57031, 90.53674, 274.01, -36.91406, 92.5602, 274.01, -39.25781, 93.55952, 274.01, -41.60156, 95.74537, 274.01, -43.94531, 98.26609, 274.01, -46.28906, 100.3701, 274.01, -47.02621, 101.1234, 274.01, -47.86611, 103.4672, 274.01, -49.83594, 105.8109, 274.01, -51.98182, 108.1547, 274.01, -53.06448, 110.4984, 274.01, -53.32031, 110.7675, 274.01, -54.53804, 112.8422, 274.01, -55.66406, 114.273, 274.01, -56.55722, 115.1859, 274.01, -57.13953, 117.5297, 274.01, -58.29264, 119.8734, 274.01, -59.26869, 122.2172, 274.01, -60.35156, 124.0119, 274.01, -60.84229, 124.5609, 274.01, -61.54484, 126.9047, 274.01, -61.71691, 129.2484, 274.01, -63.62281, 131.5922, 274.01, -63.81256, 133.9359, 274.01, -64.12511, 136.2797, 274.01, -64.84515, 138.6234, 274.01, -65.13599, 140.9672, 274.01, -65.33604, 143.3109, 274.01, -65.87358, 145.6547, 274.01, -66.10577, 147.9984, 274.01, -66.17618, 155.0297, 274.01, -66.09933, 162.0609, 274.01, -65.40382, 164.4047, 274.01, -65.24833, 166.7484, 274.01, -64.71442, 171.4359, 274.01, -63.88171, 173.7797, 274.01, -63.69299, 176.1234, 274.01, -61.79081, 178.4672, 274.01, -61.59269, 180.8109, 274.01, -61.19405, 183.1547, 274.01, -60.35156, 185.2055, 274.01, -59.08288, 187.8422, 274.01, -58.00781, 189.3499, 274.01, -57.25858, 190.1859, 274.01, -56.64558, 192.5297, 274.01, -55.29191, 194.8734, 274.01, -54.28698, 197.2172, 274.01, -52.28595, 199.5609, 274.01, -51.47569, 201.9047, 274.01, -48.63281, 204.6417, 274.01, -47.10181, 206.5922, 274.01, -44.64154, 208.9359, 274.01, -42.38504, 211.2797, 274.01, -39.25781, 214.4025, 274.01, -37.42723, 215.9672, 274.01, -34.57031, 218.9107, 274.01, -32.22656, 219.7277, 274.01, -29.88281, 221.6934, 274.01, -27.53906, 222.852, 274.01, -25.19531, 224.5168, 274.01, -22.85156, 225.9419, 274.01, -20.50781, 226.7359, 274.01, -18.16406, 228.3332, 274.01, -15.82031, 229.065, 274.01, -13.47656, 229.267, 274.01, -12.63854, 230.0297, 274.01, -11.13281, 231.9201, 274.01, -10.65505, 232.3734, 274.01, -11.13281, 232.7794, 274.01, -15.82031, 232.792, 274.01, -18.16406, 232.8902, 274.01, -36.91406, 232.9015, 274.01, -39.25781, 232.8902, 274.01, -50.97656, 232.9236, 274.01, -60.35156, 232.9126, 274.01, -67.38281, 232.8407, 274.01, -72.07031, 232.8642, 274.01, -79.10156, 232.8555, 274.01, -83.78906, 232.8788, 274.01, -95.50781, 232.8902, 274.01, -97.85156, 233.4886, 274.01, -100.1953, 233.647, 274.01, -102.5391, 232.9858, 274.01, -104.8828, 233.6969, 274.01, -109.5703, 233.722, 274.01, -125.9766, 233.7086, 274.01, -128.3203, 233.2849, 274.01, -130.6641, 233.702, 274.01, -135.3516, 233.727, 274.01, -149.4141, 233.7135, 274.01, -156.4453, 233.727, 274.01, -163.4766, 233.7119, 274.01, -168.1641, 233.7643, 274.01, -191.6016, 233.7809, 274.01, -210.3516, 233.7716, 274.01, -224.4141, 233.7998, 274.01, -233.7891, 233.7647, 274.01, -243.1641, 233.7785, 274.01, -247.8516, 233.7378, 274.01, -254.8828, 233.8578, 274.01, -257.2266, 235.2519, 274.01, -259.5703, 236.0817, 274.01, -260.7617, 237.0609, 274.01, -261.9141, 238.2262, 274.01, -262.8989, 239.4047, 274.01, -262.9743, 241.7484, 274.01, -262.5977, 244.0922, 274.01, -260.6675, 246.4359, 274.01, -259.6161, 248.7797, 274.01, -257.2266, 251.0035, 274.01, -255.0361, 253.4672, 274.01, -252.5391, 256.0995, 274.01, -251.2277, 258.1547, 274.01, -246.7444, 262.8422, 274.01, -243.1641, 266.3515, 274.01, -239.7411, 269.8734, 274.01, -238.4766, 270.9495, 274.01, -237.2269, 272.2172, 274.01, -236.1328, 273.5215, 274.01, -235.0934, 274.5609, 274.01, -233.7891, 275.6655, 274.01, -232.5319, 276.9047, 274.01, -231.4453, 278.1693, 274.01, -227.917, 281.5922, 274.01, -224.4141, 285.1802, 274.01, -222.0703, 287.4025, 274.01, -218.6841, 290.9672, 274.01, -217.3828, 291.9709, 274.01, -215.0391, 293.1788, 274.01, -212.6953, 294.5138, 274.01, -210.3516, 295.2614, 274.01, -209.8994, 295.6547, 274.01, -208.0078, 296.7083, 274.01, -203.3203, 296.9372, 274.01, -196.2891, 296.9317, 274.01, -193.9453, 296.8988, 274.01, -172.8516, 296.8482, 274.01, -161.1328, 296.843, 274.01, -137.6953, 296.8542, 274.01, -130.6641, 296.8378, 274.01, -107.2266, 296.8379, 274.01, -93.16406, 296.8208, 274.01, -74.41406, 296.838, 274.01, -65.03906, 296.8609, 274.01, -50.97656, 296.8556, 274.01, -46.28906, 296.9051, 274.01, -41.60156, 298.5331, 274.01, -39.25781, 298.5624, 274.01, -36.91406, 297.1455, 274.01, -34.57031, 297.0498, 274.01, -32.22656, 298.5589, 274.01, -25.19531, 298.5624, 274.01, -22.85156, 297.2842, 274.01, -20.50781, 298.5624, 274.01, -1.757813, 298.5624, 274.01, 0.5859375, 297.2104, 274.01, 2.929688, 298.5624, 274.01, 5.273438, 297.6946, 274.01, 7.617188, 298.5168, 274.01, 9.960938, 298.5512, 274.01, 12.30469, 296.937, 274.01, 14.64844, 298.5478, 274.01, 16.99219, 298.5478, 274.01, 19.33594, 297.0782, 274.01, 21.67969, 296.844, 274.01, 23.54531, 297.9984, 274.01, 24.02344, 298.4023, 274.01, 24.50156, 297.9984, 274.01, 26.36719, 296.844, 274.01, 38.08594, 296.8381, 274.01, 52.14844, 296.8033, 274.01, 59.17969, 296.8033, 274.01, 73.24219, 296.7682, 274.01, 99.02344, 296.7566, 274.01, 117.7734, 296.7216, 274.01, 129.4922, 296.7152, 274.01, 131.8359, 295.9083, 274.01, 134.1797, 295.5245, 274.01, 138.8672, 295.4763, 274.01, 155.2734, 295.4763, 274.01, 176.3672, 295.3861, 274.01, 190.4297, 295.3718, 274.01, 197.4609, 295.4763, 274.01, 202.1484, 295.4454, 274.01, 204.4922, 295.3438, 274.01, 206.8359, 295.0757, 274.01, 209.1797, 294.4124, 274.01, 211.5234, 292.3133, 274.01, 213.8672, 291.0809, 274.01, 216.2109, 289.6743, 274.01, 217.3081, 288.6234, 274.01, 219.3558, 286.2797, 274.01, 221.8608, 283.9359, 274.01, 225.5859, 280.045, 274.01, 227.9297, 277.8885, 274.01, 230.2734, 275.2857, 274.01, 232.6172, 273.2225, 274.01, 233.6225, 272.2172, 274.01, 234.9609, 270.5822, 274.01, 238.2254, 267.5297, 274.01, 240.3691, 265.1859, 274.01, 244.3359, 261.3326, 274.01, 246.6797, 258.8034, 274.01, 249.0234, 256.7196, 274.01, 251.3672, 254.0746, 274.01, 254.5313, 251.1234, 274.01, 255.333, 248.7797, 274.01, 257.3723, 246.4359, 274.01, 259.7201, 244.0922, 274.01, 260.106, 241.7484, 274.01, 261.6423, 239.4047, 274.01, 261.0804, 237.0609, 274.01, 259.3552, 234.7172, 274.01, 258.3984, 233.7696, 274.01, 256.0547, 232.8757, 274.01, 253.7109, 232.792, 274.01, 251.3672, 232.8161, 274.01, 246.6797, 232.6981, 274.01, 244.3359, 232.725, 274.01, 239.6484, 232.9137, 274.01, 234.9609, 232.8525, 274.01, 225.5859, 232.8757, 274.01, 209.1797, 232.8757, 274.01, 204.4922, 232.7537, 274.01, 195.1172, 232.7794, 274.01, 171.6797, 232.792, 274.01, 164.6484, 232.7666, 274.01, 152.9297, 232.7666, 274.01, 148.2422, 232.792, 274.01, 138.8672, 232.7406, 274.01 };
    size_t n = sizeof(groundTruthRaw) / sizeof(groundTruthRaw[0]); 
    std::vector<double> groundTruth(groundTruthRaw, groundTruthRaw+n);
    EXPECT_EQ(340u * 3, groundTruth.size());
    CheckGroundTruth(structures, 5, 0, groundTruth);
  }

  {
    double groundTruthRaw[] = { -18.16406, 233.0632, 298.01, -27.53906, 233.1042, 298.01, -29.88281, 233.0819, 298.01, -34.57031, 233.131, 298.01, -43.94531, 233.1221, 298.01, -50.97656, 233.1736, 298.01, -62.69531, 233.1397, 298.01, -65.03906, 232.8376, 298.01, -69.72656, 232.9839, 298.01, -79.10156, 233.0245, 298.01, -90.82031, 233.0382, 298.01, -93.16406, 233.0859, 298.01, -109.5703, 233.1132, 298.01, -111.9141, 233.1791, 298.01, -114.2578, 233.7139, 298.01, -118.9453, 233.9793, 298.01, -128.3203, 234.0284, 298.01, -130.6641, 233.9793, 298.01, -135.3516, 234.0591, 298.01, -137.6953, 234.0284, 298.01, -142.3828, 234.0855, 298.01, -144.7266, 234.0284, 298.01, -151.7578, 234.002, 298.01, -158.7891, 234.0263, 298.01, -163.4766, 233.9784, 298.01, -165.8203, 234.0072, 298.01, -168.1641, 234.1756, 298.01, -170.5078, 234.2214, 298.01, -179.8828, 234.1934, 298.01, -186.9141, 234.2721, 298.01, -189.2578, 234.2289, 298.01, -193.9453, 234.2431, 298.01, -198.6328, 234.1692, 298.01, -200.9766, 234.2326, 298.01, -205.6641, 234.1271, 298.01, -212.6953, 234.2224, 298.01, -215.0391, 234.1992, 298.01, -222.0703, 234.3115, 298.01, -224.4141, 234.2224, 298.01, -226.7578, 234.2502, 298.01, -233.7891, 234.0906, 298.01, -238.4766, 234.0329, 298.01, -243.1641, 234.0283, 298.01, -247.8516, 233.7949, 298.01, -250.1953, 233.8681, 298.01, -252.5391, 234.7626, 298.01, -254.3469, 237.0609, 298.01, -255.6034, 239.4047, 298.01, -254.5181, 241.7484, 298.01, -254.2274, 244.0922, 298.01, -254.181, 248.7797, 298.01, -253.9355, 251.1234, 298.01, -253.5926, 253.4672, 298.01, -252.7483, 255.8109, 298.01, -250.8092, 258.1547, 298.01, -248.713, 260.4984, 298.01, -246.263, 262.8422, 298.01, -244.1406, 265.1859, 298.01, -241.6671, 267.5297, 298.01, -239.4754, 269.8734, 298.01, -237.0156, 272.2172, 298.01, -233.7891, 275.382, 298.01, -231.4453, 277.8249, 298.01, -229.1016, 279.9981, 298.01, -226.7578, 282.5281, 298.01, -224.4141, 284.6784, 298.01, -222.0703, 287.2355, 298.01, -220.5414, 288.6234, 298.01, -218.2745, 290.9672, 298.01, -217.3828, 291.6508, 298.01, -212.6953, 294.5949, 298.01, -210.3516, 295.3142, 298.01, -208.0078, 296.4674, 298.01, -205.6641, 296.8852, 298.01, -203.3203, 297.1563, 298.01, -196.2891, 297.1488, 298.01, -193.9453, 297.0597, 298.01, -182.2266, 296.9529, 298.01, -168.1641, 296.8576, 298.01, -154.1016, 296.9249, 298.01, -149.4141, 296.8921, 298.01, -128.3203, 296.9228, 298.01, -121.2891, 296.8623, 298.01, -111.9141, 296.8549, 298.01, -107.2266, 296.8266, 298.01, -102.5391, 296.8731, 298.01, -95.50781, 296.8453, 298.01, -88.47656, 296.9218, 298.01, -83.78906, 296.9016, 298.01, -69.72656, 296.979, 298.01, -67.38281, 296.9514, 298.01, -65.03906, 297.2199, 298.01, -62.69531, 296.9622, 298.01, -55.66406, 296.9926, 298.01, -50.97656, 296.9467, 298.01, -48.63281, 297.3652, 298.01, -46.28906, 297.0439, 298.01, -43.94531, 297.2875, 298.01, -39.25781, 297.0121, 298.01, -34.57031, 297.1564, 298.01, -32.22656, 297.3612, 298.01, -29.88281, 297.4229, 298.01, -27.53906, 297.1687, 298.01, -25.19531, 297.4334, 298.01, -18.16406, 297.3612, 298.01, -15.82031, 297.4441, 298.01, -13.47656, 297.4125, 298.01, -11.13281, 297.2468, 298.01, -8.789063, 297.4125, 298.01, -6.445313, 297.373, 298.01, -4.101563, 297.4195, 298.01, -1.757813, 297.077, 298.01, 0.5859375, 297.4229, 298.01, 2.929688, 297.4125, 298.01, 5.273438, 296.9489, 298.01, 7.617188, 297.3168, 298.01, 9.960938, 296.9377, 298.01, 12.30469, 296.8998, 298.01, 14.64844, 297.1975, 298.01, 16.99219, 296.8579, 298.01, 28.71094, 296.878, 298.01, 40.42969, 296.8163, 298.01, 42.77344, 296.8369, 298.01, 49.80469, 296.734, 298.01, 59.17969, 296.6906, 298.01, 61.52344, 296.6365, 298.01, 68.55469, 296.6278, 298.01, 73.24219, 296.5777, 298.01, 75.58594, 296.6191, 298.01, 84.96094, 296.5284, 298.01, 96.67969, 296.5538, 298.01, 103.7109, 296.479, 298.01, 115.4297, 296.4259, 298.01, 122.4609, 296.3434, 298.01, 129.4922, 296.3495, 298.01, 131.8359, 295.9141, 298.01, 136.5234, 296.2256, 298.01, 138.8672, 295.833, 298.01, 143.5547, 295.9857, 298.01, 145.8984, 295.8791, 298.01, 152.9297, 295.833, 298.01, 164.6484, 295.6819, 298.01, 171.6797, 295.6819, 298.01, 181.0547, 295.5401, 298.01, 185.7422, 295.5742, 298.01, 192.7734, 295.557, 298.01, 197.4609, 295.8012, 298.01, 202.1484, 295.6819, 298.01, 204.4922, 295.3698, 298.01, 206.8359, 294.803, 298.01, 209.1797, 294.3656, 298.01, 211.5234, 292.4764, 298.01, 213.8672, 291.1765, 298.01, 216.2109, 289.5873, 298.01, 217.229, 288.6234, 298.01, 218.5547, 287.0752, 298.01, 221.7097, 283.9359, 298.01, 225.5859, 279.8775, 298.01, 227.9297, 277.5633, 298.01, 230.2734, 275.0808, 298.01, 233.1989, 272.2172, 298.01, 234.9609, 270.2887, 298.01, 237.7384, 267.5297, 298.01, 241.9922, 263.0843, 298.01, 244.3359, 260.7643, 298.01, 246.788, 258.1547, 298.01, 249.0234, 255.451, 298.01, 250.3651, 253.4672, 298.01, 251.5297, 251.1234, 298.01, 252.1947, 248.7797, 298.01, 252.4915, 246.4359, 298.01, 252.5755, 241.7484, 298.01, 252.8592, 239.4047, 298.01, 252.9236, 237.0609, 298.01, 252.2924, 234.7172, 298.01, 251.3672, 233.4697, 298.01, 249.0234, 232.882, 298.01, 244.3359, 232.9048, 298.01, 241.9922, 233.0145, 298.01, 232.6172, 232.9048, 298.01, 227.9297, 233.0007, 298.01, 216.2109, 233.0632, 298.01, 211.5234, 233.0537, 298.01, 206.8359, 232.9699, 298.01, 204.4922, 232.7322, 298.01, 199.8047, 232.7186, 298.01, 190.4297, 232.7719, 298.01, 183.3984, 232.7719, 298.01, 181.0547, 232.7322, 298.01, 174.0234, 232.7048, 298.01, 171.6797, 232.7322, 298.01, 166.9922, 232.6908, 298.01, 157.6172, 232.7975, 298.01, 155.2734, 232.7588, 298.01, 148.2422, 232.7875, 298.01, 143.5547, 232.7614, 298.01, 138.8672, 232.6477, 298.01, 124.8047, 232.6179, 298.01, 122.4609, 232.6477, 298.01, 113.0859, 232.6027, 298.01, 110.7422, 232.4552, 298.01, 108.3984, 232.2192, 298.01, 106.0547, 231.6764, 298.01, 103.7109, 231.8559, 298.01, 102.8237, 232.3734, 298.01, 101.3672, 232.9839, 298.01, 99.02344, 233.0951, 298.01, 87.30469, 233.0819, 298.01, 84.96094, 233.1091, 298.01, 80.27344, 233.0726, 298.01, 77.92969, 233.1132, 298.01, 70.89844, 233.1397, 298.01, 68.55469, 233.1132, 298.01, 52.14844, 233.131, 298.01, 45.11719, 233.0859, 298.01, 44.16726, 232.3734, 298.01, 42.77344, 231.0206, 298.01, 42.04498, 230.0297, 298.01, 42.77344, 229.2462, 298.01, 45.11719, 228.5664, 298.01, 47.46094, 227.0695, 298.01, 49.80469, 226.0552, 298.01, 52.14844, 224.5723, 298.01, 54.49219, 223.6857, 298.01, 56.83594, 221.8519, 298.01, 59.17969, 220.2086, 298.01, 61.52344, 218.8854, 298.01, 64.94469, 215.9672, 298.01, 66.21094, 215.0191, 298.01, 67.72036, 213.6234, 298.01, 68.55469, 212.6986, 298.01, 70.89844, 210.5055, 298.01, 74.53191, 206.5922, 298.01, 76.54903, 204.2484, 298.01, 78.26105, 201.9047, 298.01, 80.27344, 198.9262, 298.01, 82.61719, 195.2822, 298.01, 82.98087, 194.8734, 298.01, 84.96094, 190.9255, 298.01, 85.43701, 190.1859, 298.01, 86.33423, 187.8422, 298.01, 87.78722, 185.4984, 298.01, 88.60233, 183.1547, 298.01, 89.10253, 180.8109, 298.01, 90.17504, 178.4672, 298.01, 90.88959, 176.1234, 298.01, 91.43783, 173.7797, 298.01, 92.39601, 171.4359, 298.01, 92.95762, 169.0922, 298.01, 93.55695, 159.7172, 298.01, 93.65527, 157.3734, 298.01, 93.67542, 152.6859, 298.01, 93.61213, 150.3422, 298.01, 93.22542, 143.3109, 298.01, 93.06345, 140.9672, 298.01, 92.77563, 138.6234, 298.01, 91.21714, 133.9359, 298.01, 90.67235, 131.5922, 298.01, 89.88776, 129.2484, 298.01, 88.8737, 126.9047, 298.01, 88.44087, 124.5609, 298.01, 86.09712, 119.8734, 298.01, 85.05786, 117.5297, 298.01, 83.87151, 115.1859, 298.01, 82.22388, 112.8422, 298.01, 81.09117, 110.4984, 298.01, 77.92969, 106.4052, 298.01, 77.3894, 105.8109, 298.01, 75.94332, 103.4672, 298.01, 71.71799, 98.77969, 298.01, 68.55469, 95.65721, 298.01, 63.86719, 91.54878, 298.01, 61.52344, 90.1121, 298.01, 59.17969, 88.15762, 298.01, 56.83594, 86.51503, 298.01, 54.49219, 85.42721, 298.01, 52.14844, 83.64907, 298.01, 49.80469, 82.89023, 298.01, 47.46094, 81.50237, 298.01, 45.11719, 80.62591, 298.01, 42.77344, 79.18153, 298.01, 40.42969, 78.7203, 298.01, 38.08594, 78.1349, 298.01, 35.74219, 77.11755, 298.01, 33.39844, 76.51949, 298.01, 31.05469, 76.07934, 298.01, 26.36719, 74.67744, 298.01, 24.02344, 74.42056, 298.01, 14.64844, 74.07317, 298.01, 9.960938, 74.11538, 298.01, 2.929688, 74.40105, 298.01, 0.5859375, 74.67952, 298.01, -1.757813, 75.31406, 298.01, -4.101563, 76.07065, 298.01, -6.445313, 76.49051, 298.01, -8.789063, 77.17276, 298.01, -11.13281, 78.20097, 298.01, -15.82031, 79.31967, 298.01, -18.16406, 80.76948, 298.01, -20.50781, 81.64266, 298.01, -22.85156, 83.0305, 298.01, -25.19531, 83.7937, 298.01, -27.53906, 85.63515, 298.01, -29.88281, 86.7363, 298.01, -32.22656, 88.36089, 298.01, -34.57031, 90.3302, 298.01, -36.56719, 91.74844, 298.01, -41.60156, 95.93605, 298.01, -46.58845, 101.1234, 298.01, -50.17995, 105.8109, 298.01, -52.10386, 108.1547, 298.01, -53.63992, 110.4984, 298.01, -54.95532, 112.8422, 298.01, -56.64794, 115.1859, 298.01, -57.4403, 117.5297, 298.01, -58.91927, 119.8734, 298.01, -59.78655, 122.2172, 298.01, -61.11754, 124.5609, 298.01, -61.58921, 126.9047, 298.01, -62.38012, 129.2484, 298.01, -63.49118, 131.5922, 298.01, -64.02599, 133.9359, 298.01, -64.3932, 136.2797, 298.01, -65.11897, 138.6234, 298.01, -65.64544, 140.9672, 298.01, -66.23938, 147.9984, 298.01, -66.46289, 152.6859, 298.01, -66.48911, 155.0297, 298.01, -66.34437, 159.7172, 298.01, -65.99894, 164.4047, 298.01, -65.49149, 169.0922, 298.01, -64.6875, 171.4359, 298.01, -63.7739, 176.1234, 298.01, -62.9398, 178.4672, 298.01, -61.86011, 180.8109, 298.01, -61.33423, 183.1547, 298.01, -60.43332, 185.4984, 298.01, -58.00781, 190.0632, 298.01, -56.85406, 192.5297, 298.01, -55.66406, 194.7283, 298.01, -54.11692, 197.2172, 298.01, -50.97656, 201.8369, 298.01, -47.36435, 206.5922, 298.01, -45.04395, 208.9359, 298.01, -42.83026, 211.2797, 298.01, -39.25781, 214.7435, 298.01, -34.57031, 218.4974, 298.01, -32.22656, 219.9595, 298.01, -28.02053, 222.9984, 298.01, -27.53906, 223.4238, 298.01, -25.19531, 224.4187, 298.01, -22.85156, 225.8252, 298.01, -20.50781, 226.9067, 298.01, -18.16406, 228.4286, 298.01, -15.82031, 229.1235, 298.01, -14.9447, 230.0297, 298.01, -15.82031, 231.3969, 298.01, -16.94484, 232.3734, 298.01 };
    size_t n = sizeof(groundTruthRaw) / sizeof(groundTruthRaw[0]); 
    std::vector<double> groundTruth(groundTruthRaw, groundTruthRaw+n);
    EXPECT_EQ(358u * 3, groundTruth.size());
    CheckGroundTruth(structures, 5, 8, groundTruth);
  }
}

#endif 
// BGO_ENABLE_DICOMSTRUCTURESETLOADER2

#if 0

TEST(StructureSet2, ReadFromJsonAndCompute1)
{
  DicomStructureSet2 structureSet;

  OrthancPlugins::FullOrthancDataset dicom(GetTestJson());
  structureSet.Clear();

  structureSet.FillStructuresFromDataset(dicom);

  structureSet.ComputeDependentProperties();
}

TEST(StructureSet2, ReadFromJsonAndCompute2)
{
  DicomStructureSet2 structureSet;

  OrthancPlugins::FullOrthancDataset dicom(GetTestJson());
  structureSet.Clear();

  structureSet.SetContents(dicom);
}
#endif

#ifdef BGO_ENABLE_DICOMSTRUCTURESETLOADER2

static bool CutStructureWithPlane(
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> >& segments,
  const DicomStructure2& structure,
  const double originX, const double originY, const double originZ,
  const double axisX_X, const double axisX_Y, const double axisX_Z,
  const double axisY_X, const double axisY_Y, const double axisY_Z
)
{
  // create an AXIAL cutting plane, too far away from the volume 
  // (> sliceThickness/2)
  Vector origin, axisX, axisY;
  LinearAlgebra::AssignVector(origin, originX, originY, originZ);
  LinearAlgebra::AssignVector(axisX, axisX_X, axisX_Y, axisX_Z);
  LinearAlgebra::AssignVector(axisY, axisY_X, axisY_Y, axisY_Z);
  CoordinateSystem3D cuttingPlane(origin, axisX, axisY);

  // compute intersection
  bool ok = structure.Project(segments, cuttingPlane);
  return ok;
}

#endif 
// BGO_ENABLE_DICOMSTRUCTURESETLOADER2

#ifdef BGO_ENABLE_DICOMSTRUCTURESETLOADER2

static void CreateBasicStructure(DicomStructure2& structure)
{
  // see https://www.dropbox.com/s/1o1vg53hsbvx4cc/test-rtstruct-polygons.jpg?dl=0
  EXPECT_EQ(pointsCoord1Count, pointsCoord2Count);
  EXPECT_EQ(11u, pointsCoord2Count);

  for (size_t slice = 0; slice < 3; ++slice)
  {
    DicomStructurePolygon2 polygon("Oblomptu", "CLOSED_PLANAR");
    for (size_t ip = 0; ip < pointsCoord1Count; ++ip)
    {
      Vector pt;
      double pt0 = pointsCoord1[ip];
      double pt1 = pointsCoord2[ip];
      double pt2 = 4 * (static_cast<double>(slice) - 1); // -4, 0, 4 
      LinearAlgebra::AssignVector(pt, pt0, pt1, pt2);
      polygon.AddPoint(pt);
    }
    structure.AddPolygon(polygon);
  }
  structure.ComputeDependentProperties();
}


TEST(StructureSet2, CutAxialOutsideTop)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  // create an AXIAL cutting plane, too far away from the volume 
  // (> sliceThickness/2)
  bool ok = CutStructureWithPlane(segments, structure,
    0, 0, 7,
    1, 0, 0,
    0, 1, 0);
  EXPECT_FALSE(ok);
}


TEST(StructureSet2, CutAxialOutsideBottom)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  // create an AXIAL cutting plane, too far away from the volume 
  // (> sliceThickness/2)
  bool ok = CutStructureWithPlane(segments, structure,
    0, 0, -6.66,
    1, 0, 0,
    0, 1, 0);
  EXPECT_FALSE(ok);
}

TEST(StructureSet2, CutAxialInsideClose)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  // create an AXIAL cutting plane in the volume
  bool ok = CutStructureWithPlane(segments, structure,
    0, 0, 1.1,
    1, 0, 0,
    0, 1, 0);
  EXPECT_TRUE(ok);
  EXPECT_EQ(POLYGON_POINT_COUNT, segments.size());

  for (size_t i = 0; i < segments.size(); ++i)
  {
    EXPECT_LT(i, POLYGON_POINT_COUNT);
    EXPECT_LT(i, POLYGON_POINT_COUNT);

    const ScenePoint2D& pt = segments[i].first;

    // ...should be at the same location as the 3D coords since the plane 
    // is rooted at 0,0,0 with normal 0,0,1
    EXPECT_NEAR(pt.GetX(), pointsCoord1[i], DELTA_MAX);
    EXPECT_NEAR(pt.GetY(), pointsCoord2[i], DELTA_MAX);
  }
}


TEST(StructureSet2, CutAxialInsideFar)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  // create an AXIAL cutting plane
  Vector origin, axisX, axisY;
  LinearAlgebra::AssignVector(origin, 0, 0, 0);
  LinearAlgebra::AssignVector(axisX, 1, 0, 0);
  LinearAlgebra::AssignVector(axisY, 0, 1, 0);
  CoordinateSystem3D cuttingPlane(origin, axisX, axisY);

  // compute intersection
  bool ok = structure.Project(segments, cuttingPlane);
  EXPECT_TRUE(ok);

  EXPECT_EQ(11u, segments.size());
  for (size_t i = 0; i < segments.size(); ++i)
  {
    EXPECT_LT(i, pointsCoord1Count);
    EXPECT_LT(i, pointsCoord2Count);

    // the 2D points of the projected polygon
    const ScenePoint2D& pt = segments[i].first;

    // ...should be at the same location as the 3D coords since the plane 
    // is rooted at 0,0,0 with normal 0,0,1
    EXPECT_NEAR(pt.GetX(), pointsCoord1[i], DELTA_MAX);
    EXPECT_NEAR(pt.GetY(), pointsCoord2[i], DELTA_MAX);
  }
}

TEST(StructureSet2, CutCoronalOutsideClose)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  // create an X,Z cutting plane, outside of the volume
  Vector origin, axisX, axisY;
  LinearAlgebra::AssignVector(origin, 0, 0, 0);
  LinearAlgebra::AssignVector(axisX, 1, 0, 0);
  LinearAlgebra::AssignVector(axisY, 0, 0, 1);
  CoordinateSystem3D cuttingPlane(origin, axisX, axisY);

  // compute intersection
  bool ok = structure.Project(segments, cuttingPlane);
  EXPECT_FALSE(ok);
}

TEST(StructureSet2, CutCoronalInsideClose1DTest)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);

  // create an X,Z cutting plane, outside of the volume
  Vector origin, axisX, axisY;
  LinearAlgebra::AssignVector(origin, 0, 3, 0);
  LinearAlgebra::AssignVector(axisX, 1, 0, 0);
  LinearAlgebra::AssignVector(axisY, 0, 0, 1);
  CoordinateSystem3D cuttingPlane(origin, axisX, axisY);

  ASSERT_EQ(3u, structure.GetPolygons().size());

  for (int i = 0; i < 3; ++i)
  {
    double polygonZ = static_cast<double>(i - 1) * 4.0;

    const DicomStructurePolygon2& topSlab = structure.GetPolygons()[i];

    // let's compute the intersection between the polygon and the plane
    // intersections are in plane coords
    std::vector<ScenePoint2D> intersects;
    topSlab.ProjectOnConstantPlane(intersects, cuttingPlane);

    ASSERT_EQ(4u, intersects.size());

    EXPECT_NEAR(2, intersects[0].GetX(), DELTA_MAX);
    EXPECT_NEAR(4, intersects[1].GetX(), DELTA_MAX);
    EXPECT_NEAR(7, intersects[2].GetX(), DELTA_MAX);
    EXPECT_NEAR(8, intersects[3].GetX(), DELTA_MAX);

    for (size_t i = 0; i < 4u; ++i)
    {
      EXPECT_NEAR(polygonZ, intersects[i].GetY(), DELTA_MAX);
    }
  }
}

TEST(StructureSet2, CutCoronalInsideClose)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  // create an X,Z cutting plane, outside of the volume
  Vector origin, axisX, axisY;
  LinearAlgebra::AssignVector(origin, 0, 3, 0);
  LinearAlgebra::AssignVector(axisX, 1, 0, 0);
  LinearAlgebra::AssignVector(axisY, 0, 0, 1);
  CoordinateSystem3D cuttingPlane(origin, axisX, axisY);

  // compute intersection
  ASSERT_TRUE(structure.Project(segments, cuttingPlane));
  EXPECT_EQ(24u, segments.size());

  size_t numberOfVeryShortSegments = 0;
  for (size_t iSegment = 0; iSegment < segments.size(); ++iSegment)
  {
    // count the NON vertical very short segments 
    if (LinearAlgebra::IsNear(segments[iSegment].first.GetX(), segments[iSegment].second.GetX()))
    {
      if (LinearAlgebra::IsNear(segments[iSegment].first.GetY(), segments[iSegment].second.GetY()))
      {
        numberOfVeryShortSegments++;
      }
    }
  }
  EXPECT_EQ(8u, numberOfVeryShortSegments);
}

#endif 
// BGO_ENABLE_DICOMSTRUCTURESETLOADER2

TEST(DisjointDataSet, BasicTest)
{
  const size_t ITEM_COUNT = 10;
  DisjointDataSet ds(ITEM_COUNT);

  for (size_t i = 0; i < ITEM_COUNT; ++i)
  {
    EXPECT_EQ(i, ds.Find(i));
  }

  ds.Union(0, 4);
  EXPECT_EQ(0u, ds.Find(0));
  EXPECT_EQ(0u, ds.Find(4));

  ds.Union(4, 6);
  ds.Union(8, 9);
  ds.Union(0, 8);

  for (size_t i = 0; i < ITEM_COUNT; ++i)
  {
    size_t parent = ds.Find(i);
    EXPECT_TRUE(0 == parent || 1 == parent || 2 == parent || 3 == parent || 5 == parent || 7 == parent);
  }

  ds.Union(1, 2);
  ds.Union(1, 7);
  for (size_t i = 0; i < ITEM_COUNT; ++i)
  {
    size_t parent = ds.Find(i);
    EXPECT_TRUE(0 == parent || 1 == parent || 3 == parent || 5 == parent);
  }

  ds.Union(3, 5);
  for (size_t i = 0; i < ITEM_COUNT; ++i)
  {
    size_t parent = ds.Find(i);
    EXPECT_TRUE(0 == parent || 1 == parent || 3 == parent);
  }

  EXPECT_EQ(ds.Find(0), ds.Find(0));
  EXPECT_EQ(ds.Find(0), ds.Find(4));
  EXPECT_EQ(ds.Find(0), ds.Find(6));
  EXPECT_EQ(ds.Find(0), ds.Find(8));
  EXPECT_EQ(ds.Find(0), ds.Find(8));

  EXPECT_EQ(ds.Find(1), ds.Find(7));
  EXPECT_EQ(ds.Find(2), ds.Find(1));
  EXPECT_EQ(ds.Find(7), ds.Find(2));

  EXPECT_EQ(ds.Find(3), ds.Find(5));
  EXPECT_EQ(ds.Find(5), ds.Find(3));

  ds.Union(0, 1);
  ds.Union(3, 1);
  for (size_t i = 0; i < ITEM_COUNT; ++i)
  {
    EXPECT_EQ(ds.Find(0), ds.Find(i));
  }
}

#ifdef BGO_ENABLE_DICOMSTRUCTURESETLOADER2

TEST(StructureSet2, CutSagittalInsideClose)
{
  DicomStructure2 structure;
  CreateBasicStructure(structure);
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  // create an X,Z cutting plane, inside of the volume
  Vector origin, axisX, axisY;
  LinearAlgebra::AssignVector(origin, 0, 3, 0);
  LinearAlgebra::AssignVector(axisX, 1, 0, 0);
  LinearAlgebra::AssignVector(axisY, 0, 0, 1);
  CoordinateSystem3D cuttingPlane(origin, axisX, axisY);

  // compute intersection
  bool ok = structure.Project(segments, cuttingPlane);
  EXPECT_TRUE(ok);
}

#endif 
// BGO_ENABLE_DICOMSTRUCTURESETLOADER2


static size_t ConvertListOfSlabsToSegments_Add(RtStructRectanglesInSlab& rectangles, int row, double xmin, double xmax)
{
  double ymin = static_cast<double>(row) * 5.0;
  double ymax = static_cast<double>(row + 1) * 5.0;

  RtStructRectangleInSlab rectangle;
  rectangle.xmin = xmin;
  rectangle.xmax = xmax;
  rectangle.ymin = ymin;
  rectangle.ymax = ymax;

  rectangles.push_back(rectangle);
  
  return 1u;
}

static size_t FillTestRectangleList(std::vector< RtStructRectanglesInSlab >& rectanglesForEachSlab)
{
  // ConvertListOfSlabsToSegments
  size_t rectCount = 0;

  rectanglesForEachSlab.push_back(RtStructRectanglesInSlab());
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 0, 5, 31);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 0, 36, 50);

  rectanglesForEachSlab.push_back(RtStructRectanglesInSlab());
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 1, 20, 45);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 1, 52, 70);

  rectanglesForEachSlab.push_back(RtStructRectanglesInSlab());
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 2, 0, 32);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 2, 35, 44);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 2, 60, 75);

  rectanglesForEachSlab.push_back(RtStructRectanglesInSlab());
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 3, 10, 41);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 3, 46, 80);

  rectanglesForEachSlab.push_back(RtStructRectanglesInSlab());
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 4, 34, 42);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 4, 90, 96);

  rectanglesForEachSlab.push_back(RtStructRectanglesInSlab());
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 5, 1, 33);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 5, 40, 43);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 5, 51, 61);
  rectCount += ConvertListOfSlabsToSegments_Add(rectanglesForEachSlab.back(), 5, 76, 95);

  return rectCount;
}

/*
void AddSlabBoundaries(
  std::vector<std::pair<double, RectangleBoundaryKind> >& boundaries,
  const std::vector<RtStructRectanglesInSlab>& slabCuts, size_t iSlab)
*/


/*
void ProcessBoundaryList(
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> >& segments,
  const std::vector<std::pair<double, RectangleBoundaryKind> >& boundaries,
  double y)
*/


TEST(StructureSet2, ProcessBoundaryList_Empty)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;

  boundaries.clear();
  EXPECT_NO_THROW(AddSlabBoundaries(boundaries, slabCuts, 0));
  ASSERT_EQ(0u, boundaries.size());
}

TEST(StructureSet2, ProcessBoundaryListTopRow)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
  FillTestRectangleList(slabCuts);

  boundaries.clear();
  AddSlabBoundaries(boundaries, slabCuts, 0);

  {
    size_t i = 0;
    ASSERT_EQ(4u, boundaries.size());

    ASSERT_EQ(RectangleBoundaryKind_Start, boundaries[i].second);
    ASSERT_NEAR(5, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_End, boundaries[i].second);
    ASSERT_NEAR(31, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_Start, boundaries[i].second);
    ASSERT_NEAR(36, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_End, boundaries[i].second);
    ASSERT_NEAR(50, boundaries[i].first, DELTA_MAX);
    i++;
  }
}

TEST(StructureSet2, ProcessBoundaryListRows_0_and_1)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
  FillTestRectangleList(slabCuts);

  boundaries.clear();
  AddSlabBoundaries(boundaries, slabCuts, 0);
  AddSlabBoundaries(boundaries, slabCuts, 1);

  ASSERT_EQ(8u, boundaries.size());

  {
    size_t i = 0;

    ASSERT_EQ(RectangleBoundaryKind_Start, boundaries[i].second);
    ASSERT_NEAR(5, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_Start, boundaries[i].second);
    ASSERT_NEAR(20, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_End, boundaries[i].second);
    ASSERT_NEAR(31, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_Start, boundaries[i].second);
    ASSERT_NEAR(36, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_End, boundaries[i].second);
    ASSERT_NEAR(45, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_End, boundaries[i].second);
    ASSERT_NEAR(50, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_Start, boundaries[i].second);
    ASSERT_NEAR(52, boundaries[i].first, DELTA_MAX);
    i++;

    ASSERT_EQ(RectangleBoundaryKind_End, boundaries[i].second);
    ASSERT_NEAR(70, boundaries[i].first, DELTA_MAX);
    i++;
  }
}

TEST(StructureSet2, ConvertListOfSlabsToSegments_EmptyBoundaries)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
  FillTestRectangleList(slabCuts);
  boundaries.clear();
  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;
  ASSERT_NO_THROW(ProcessBoundaryList(segments, boundaries, 42.0));
  ASSERT_EQ(0u, segments.size());
}

TEST(StructureSet2, ConvertListOfSlabsToSegments_TopRow_Horizontal)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  FillTestRectangleList(slabCuts);

  // top row
  {
    std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;
    std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
    AddSlabBoundaries(boundaries, slabCuts, 0);
    ProcessBoundaryList(segments, boundaries, slabCuts[0][0].ymin);

    ASSERT_EQ(2u, segments.size());

    ASSERT_NEAR( 5.0, segments[0].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(31.0, segments[0].second.GetX(), DELTA_MAX);
    ASSERT_NEAR( 0.0, segments[0].first.GetY(), DELTA_MAX);
    ASSERT_NEAR( 0.0, segments[0].second.GetY(), DELTA_MAX);

    ASSERT_NEAR(36.0, segments[1].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(50.0, segments[1].second.GetX(), DELTA_MAX);
    ASSERT_NEAR( 0.0, segments[1].first.GetY(), DELTA_MAX);
    ASSERT_NEAR( 0.0, segments[1].second.GetY(), DELTA_MAX);
  }
}

TEST(StructureSet2, ConvertListOfSlabsToSegments_All_Horizontal)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
  FillTestRectangleList(slabCuts);

  // top row
  {
    std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
    AddSlabBoundaries(boundaries, slabCuts, 0);
    std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;
    ProcessBoundaryList(segments, boundaries, slabCuts[0][0].ymin);
  }

  // mids
  {
    std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
    AddSlabBoundaries(boundaries, slabCuts, 0);
    AddSlabBoundaries(boundaries, slabCuts, 1);
    std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;
    ProcessBoundaryList(segments, boundaries, slabCuts[0][0].ymax);

    ASSERT_EQ(4u, segments.size());

    ASSERT_NEAR(05.0, segments[0].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(20.0, segments[0].second.GetX(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[0].first.GetY(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[0].second.GetY(), DELTA_MAX);

    ASSERT_NEAR(31.0, segments[1].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(36.0, segments[1].second.GetX(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[1].first.GetY(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[1].second.GetY(), DELTA_MAX);

    ASSERT_NEAR(45.0, segments[2].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(50.0, segments[2].second.GetX(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[2].first.GetY(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[2].second.GetY(), DELTA_MAX);

    ASSERT_NEAR(52.0, segments[3].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(70.0, segments[3].second.GetX(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[3].first.GetY(), DELTA_MAX);
    ASSERT_NEAR(05.0, segments[3].second.GetY(), DELTA_MAX);
  }

  // bottom row
  {
    std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
    AddSlabBoundaries(boundaries, slabCuts, 1);
    std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;
    ProcessBoundaryList(segments, boundaries, slabCuts[1][0].ymax);

    ASSERT_EQ(2u, segments.size());

    ASSERT_NEAR(20.0, segments[0].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(45.0, segments[0].second.GetX(), DELTA_MAX);
    ASSERT_NEAR(10.0, segments[0].first.GetY(), DELTA_MAX);
    ASSERT_NEAR(10.0, segments[0].second.GetY(), DELTA_MAX);

    ASSERT_NEAR(52.0, segments[1].first.GetX(), DELTA_MAX);
    ASSERT_NEAR(70.0, segments[1].second.GetX(), DELTA_MAX);
    ASSERT_NEAR(10.0, segments[1].first.GetY(), DELTA_MAX);
    ASSERT_NEAR(10.0, segments[1].second.GetY(), DELTA_MAX);
  }

}

TEST(StructureSet2, ConvertListOfSlabsToSegments_Complete_Empty)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;

  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  ASSERT_NO_THROW(ConvertListOfSlabsToSegments(segments, slabCuts, 0));
  ASSERT_EQ(0u, segments.size());
}

TEST(StructureSet2, ConvertListOfSlabsToSegments_Complete_Regular)
{
  std::vector< RtStructRectanglesInSlab > slabCuts;
  std::vector<std::pair<double, RectangleBoundaryKind> > boundaries;
  size_t totalRectCount = FillTestRectangleList(slabCuts);

  std::vector< std::pair<ScenePoint2D, ScenePoint2D> > segments;

  ASSERT_NO_THROW(ConvertListOfSlabsToSegments(segments, slabCuts, totalRectCount));
  ASSERT_EQ(60u, segments.size());

  size_t i = 0;

  ASSERT_NEAR(segments[i].first.GetX(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 31.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 31.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 36.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 36.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 50.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 50.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 45.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 45.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 52.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 52.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 70.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 70.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 32.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 32.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 35.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 35.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 44.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 44.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 60.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 60.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 75.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 75.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 41.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 41.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 46.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 46.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 80.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 80.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 34.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 34.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 42.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 42.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 90.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 90.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 96.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 96.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 1.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 1.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 33.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 33.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 40.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 40.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 43.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 43.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 51.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 51.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 61.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 61.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 76.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 76.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 95.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 95.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 31.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 0.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 36.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 50.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 0.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 31.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 36.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 45.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 50.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 52.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 5.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 70.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 5.0000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 32.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 35.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 44.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 45.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 52.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 60.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 70.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 75.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 10.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 0.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 32.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 35.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 41.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 44.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 46.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 60.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 75.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 15.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 80.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 15.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 10.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 34.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 41.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 42.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 46.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 80.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 90.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 20.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 96.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 20.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 1.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 33.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 34.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 40.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 42.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 43.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 51.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 61.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 76.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 90.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 95.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 25.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 96.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 25.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 1.0000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 30.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 33.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 40.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 30.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 43.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 51.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 30.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 61.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
  i++;
  ASSERT_NEAR(segments[i].first.GetX(), 76.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].first.GetY(), 30.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetX(), 95.000000000000000, DELTA_MAX);
  ASSERT_NEAR(segments[i].second.GetY(), 30.000000000000000, DELTA_MAX);
}

#if defined(BGO_ENABLE_DICOMSTRUCTURESETLOADER2) && (ORTHANC_SANDBOXED != 1)

#include <SystemToolbox.h>

TEST(StructureSet2, ReadFromJsonPart2)
{
  DicomStructureSet2 structureSet;
  std::string jsonText;

  Orthanc::SystemToolbox::ReadFile(jsonText, "72c773ac-5059f2c4-2e6a9120-4fd4bca1-45701661.json");

  FullOrthancDataset dicom(jsonText);
  structureSet.Clear();

  structureSet.FillStructuresFromDataset(dicom);
  structureSet.ComputeDependentProperties();

  //const std::vector<DicomStructure2>& structures = structureSet.structures_;
}

#endif 
// BGO_ENABLE_DICOMSTRUCTURESETLOADER2


